import React, { useState, useEffect } from 'react';

interface TypingTextProps {
  text: string;
  className?: string;
  speed?: number; // ms per character
  cursor?: boolean;
}

export default function TypingText({ text, className = '', speed = 80, cursor = true }: TypingTextProps) {
  const [displayed, setDisplayed] = useState('');
  const [showCursor, setShowCursor] = useState(true);
  const [typingDone, setTypingDone] = useState(false);

  useEffect(() => {
    let cancelled = false;
    setDisplayed('');
    setTypingDone(false);
    function type(i: number) {
      if (cancelled) return;
      if (i <= text.length) {
        setDisplayed(text.slice(0, i));
        if (i < text.length) {
          setTimeout(() => type(i + 1), speed);
        } else {
          setTypingDone(true);
        }
      }
    }
    type(1);
    return () => { cancelled = true; };
  }, [text, speed]);

  useEffect(() => {
    if (!cursor || typingDone) return;
    const blink = setInterval(() => setShowCursor((v) => !v), 500);
    return () => clearInterval(blink);
  }, [cursor, typingDone]);

  return (
    <span className={className} style={{ whiteSpace: 'pre', position: 'static', display: 'inline-block' }}>
      {displayed}
      {cursor && !typingDone && (
        <span
          style={{
            opacity: showCursor ? 1 : 0,
            transition: 'opacity 0.2s',
            position: 'absolute',
            right: 0,
            top: 0,
            height: '100%',
            display: 'inline-block',
            pointerEvents: 'none',
          }}
        >
          |
        </span>
      )}
    </span>
  );
} 